import os
from flask import Flask, render_template, request, redirect, url_for, flash, send_from_directory
from flask_sqlalchemy import SQLAlchemy
from flask_login import LoginManager, login_user, logout_user, login_required, current_user, UserMixin
from flask_wtf import FlaskForm
from flask_wtf.file import FileField, FileAllowed, FileRequired
from wtforms import StringField, TextAreaField, PasswordField, SubmitField, BooleanField
from wtforms.validators import DataRequired
from werkzeug.security import generate_password_hash, check_password_hash
from werkzeug.utils import secure_filename
from PIL import Image
import imageio

# ----------------------------
# إعدادات التطبيق
# ----------------------------

app = Flask(__name__)
app.config.from_object('config.Config')


# إنشاء مجلدات الرفع إذا لم تكن موجودة
os.makedirs(os.path.join(app.config['UPLOAD_FOLDER'], 'products'), exist_ok=True)
os.makedirs(os.path.join(app.config['UPLOAD_FOLDER'], 'brands'), exist_ok=True)
os.makedirs(os.path.join(app.config['UPLOAD_FOLDER'], 'hero'), exist_ok=True)
os.makedirs(os.path.join(app.config['UPLOAD_FOLDER'], 'videos'), exist_ok=True)


# ----------------------------
# قاعدة البيانات والنماذج
# ----------------------------

db = SQLAlchemy(app)

class User(UserMixin, db.Model):
    id = db.Column(db.Integer, primary_key=True)
    username = db.Column(db.String(80), unique=True, nullable=False)
    password_hash = db.Column(db.String(128), nullable=False)

    def set_password(self, password):
        self.password_hash = generate_password_hash(password)

    def check_password(self, password):
        return check_password_hash(self.password_hash, password)

class Brand(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(100), nullable=False)
    logo = db.Column(db.String(200))
    products = db.relationship('Product', backref='brand', lazy=True, cascade='all, delete-orphan')

class Product(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(100), nullable=False)
    description = db.Column(db.Text)
    image = db.Column(db.String(200))
    brand_id = db.Column(db.Integer, db.ForeignKey('brand.id'), nullable=False)

class Message(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(100), nullable=False)
    subject = db.Column(db.String(200))
    body = db.Column(db.Text, nullable=False)
    read = db.Column(db.Boolean, default=False)

# --- نموذج صور الهيدر ---
class HeroImage(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    filename = db.Column(db.String(200), nullable=False)
    caption = db.Column(db.String(200))
    active = db.Column(db.Boolean, default=True)  # لتمكين/تعطيل الصورة في العرض

class GalleryImage(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    filename = db.Column(db.String(200), nullable=False)
    caption = db.Column(db.String(200))  # اختياري
    active = db.Column(db.Boolean, default=True)  # لتمكين/تعطيل الصورة

class Video(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    filename = db.Column(db.String(200), nullable=False)
    title = db.Column(db.String(200), nullable=False)
    active = db.Column(db.Boolean, default=True)


# ----------------------------
# إدارة تسجيل الدخول
# ----------------------------

login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'admin_login'

@login_manager.user_loader
def load_user(user_id):
    return User.query.get(int(user_id))

# ----------------------------
# نماذج النماذج (Forms)
# ----------------------------

class LoginForm(FlaskForm):
    username = StringField('اسم المستخدم', validators=[DataRequired()])
    password = PasswordField('كلمة المرور', validators=[DataRequired()])
    submit = SubmitField('تسجيل الدخول')

class ContactForm(FlaskForm):
    name = StringField('الاسم', validators=[DataRequired()])
    subject = StringField('الموضوع')
    body = TextAreaField('الرسالة', validators=[DataRequired()])
    submit = SubmitField('إرسال')

class ProductForm(FlaskForm):
    name = StringField('اسم المنتج', validators=[DataRequired()])
    description = TextAreaField('الوصف')
    image = FileField('صورة المنتج', validators=[FileAllowed(['png', 'jpg', 'jpeg', 'webp'], 'صور فقط!')])
    brand_id = StringField('معرف العلامة التجارية', validators=[DataRequired()])
    submit = SubmitField('حفظ')

class BrandForm(FlaskForm):
    name = StringField('اسم العلامة', validators=[DataRequired()])
    logo = FileField('شعار العلامة', validators=[FileAllowed(['png', 'jpg', 'jpeg', 'webp'], 'صور فقط!')])
    submit = SubmitField('حفظ')

# --- نموذج إضافة صور الهيدر ---
class AddHeroImageForm(FlaskForm):
    image = FileField('صورة الهيدر', validators=[FileRequired(), FileAllowed(['png', 'jpg', 'jpeg', 'webp'], 'صور فقط!')])
    caption = StringField('التعليق (اختياري)')
    active = BooleanField('مفعلة في العرض')
    submit = SubmitField('حفظ')

class EditHeroImageForm(FlaskForm):
    image = FileField('صورة الهيدر', validators=[FileAllowed(['png', 'jpg', 'jpeg', 'webp'], 'صور فقط!')])
    caption = StringField('التعليق (اختياري)')
    active = BooleanField('مفعلة في العرض')
    submit = SubmitField('تحديث')

class GalleryImageForm(FlaskForm):
    image = FileField('صورة المعرض', validators=[ FileAllowed(['png', 'jpg', 'jpeg', 'webp'], 'صور فقط!')])
    caption = StringField('التعليق (اختياري)')
    active = BooleanField('مفعلة في العرض')
    submit = SubmitField('حفظ')

class VideoForm(FlaskForm):
    title = StringField('عنوان الفيديو', validators=[DataRequired()])
    video = FileField('ملف الفيديو', validators=[FileAllowed(['mp4', 'mov', 'avi', 'mkv'], 'فيديو فقط!')])
    active = BooleanField('مفعل')
    submit = SubmitField('حفظ')

class AdminForm(FlaskForm):
    username = StringField('اسم المستخدم', validators=[DataRequired()])
    password = PasswordField('كلمة المرور', validators=[DataRequired()])
    submit = SubmitField('حفظ')
# ----------------------------
# وظائف مساعدة
# ----------------------------

def allowed_file(filename):
    return '.' in filename and \
           filename.rsplit('.', 1)[1].lower() in app.config['ALLOWED_EXTENSIONS']
def allowed_video_file(filename):
    return '.' in filename and \
           filename.rsplit('.', 1)[1].lower() in app.config.get('ALLOWED_VIDEO_EXTENSIONS', set())





def get_video_duration(input_path):
    """يحسب مدة الفيديو بالثواني"""
    reader = imageio.get_reader(input_path)
    meta = reader.get_meta_data()
    duration = meta.get('duration')
    if duration is None:
        duration = meta['nframes'] / meta['fps']
    reader.close()
    return duration

def compress_video_to_half_size(input_path, output_path):
    """
    تُرجع tuple: (نجاح: bool, رسالة: str)
    """
    try:
        original_size = os.path.getsize(input_path)
        target_size = original_size // 2
        duration = get_video_duration(input_path)
        fps = imageio.get_reader(input_path).get_meta_data()['fps']

        target_bitrate_kbps = max(200, int((target_size * 8) / (duration * 1000)))

        reader = imageio.get_reader(input_path)
        writer = imageio.get_writer(
            output_path,
            fps=fps,
            bitrate=f'{target_bitrate_kbps}k',
            codec='libx264',
            format='FFMPEG',
            ffmpeg_log_level='quiet'
        )

        for frame in reader:
            writer.append_data(frame)

        writer.close()
        reader.close()

        if os.path.exists(output_path) and os.path.getsize(output_path) > 0:
            os.remove(input_path)
            final_size = os.path.getsize(output_path)
            reduction = (1 - final_size / original_size) * 100
            msg = f"تم ضغط الفيديو بنجاح! الحجم المخفض: {reduction:.1f}%"
            return True, msg
        else:
            if os.path.exists(output_path):
                os.remove(output_path)
            return False, "فشل إنشاء الفيديو المضغوط."

    except Exception as e:
        return False, f"فشل ضغط الفيديو: {str(e)}"
    


def save_video(file, folder):
    """
    تُرجع tuple: (filename: str, message: str or None)
    """
    filename = secure_filename(file.filename)
    original_path = os.path.join(app.config['UPLOAD_FOLDER'], folder, filename)
    file.save(original_path)

    name, ext = os.path.splitext(filename)
    compressed_filename = f"{name}_compressed{ext}"
    compressed_path = os.path.join(app.config['UPLOAD_FOLDER'], folder, compressed_filename)

    success, msg = compress_video_to_half_size(original_path, compressed_path)
    if success:
        return compressed_filename, msg
    else:
        # احتفظ بالأصلي
        return filename, None  # لا نعرض رسالة خطأ هنا (قد لا يكون خطأ فعليًا)
    

def save_image(file, folder):
    if file and allowed_file(file.filename):
        filename = secure_filename(file.filename)
        path = os.path.join(app.config['UPLOAD_FOLDER'], folder, filename)
        counter = 1
        while os.path.exists(path):
            name, ext = os.path.splitext(filename)
            filename = f"{name}_{counter}{ext}"
            path = os.path.join(app.config['UPLOAD_FOLDER'], folder, filename)
            counter += 1
        file.save(path)
        try:
            img = Image.open(path)
            img.thumbnail((800, 800))
            img.save(path, optimize=True, quality=75)
        except Exception as e:
            print(f"Warning: Could not optimize image: {e}")
        return os.path.join('uploads', folder, filename).replace('\\', '/')
    return None



# ----------------------------
# المسارات (Routes)
# ----------------------------

@app.route('/')
def index():
    hero_images = HeroImage.query.filter_by(active=True).all()
    gallery_images = GalleryImage.query.filter_by(active=True).all()

    videos = Video.query.filter_by(active=True).all()
    return render_template('index.html', hero_images=hero_images, gallery_images=gallery_images, videos=videos)

@app.route('/agents')
def agents():
    return render_template('agents.html')

@app.route('/products')
def products():
    brand_id = request.args.get('brand')

    brands = Brand.query.all()

    if brand_id:
        products = Product.query.filter_by(brand_id=brand_id).all()
    else:
        products = Product.query.all()

    return render_template('products.html',
                           brands=brands,
                           products=products,
                           selected_brand=brand_id)


@app.route('/about')
def about():
    return render_template('about.html')

@app.route('/contact', methods=['GET', 'POST'])
def contact():
    form = ContactForm()
    if form.validate_on_submit():
        msg = Message(
            name=form.name.data,
            subject=form.subject.data,
            body=form.body.data
        )
        db.session.add(msg)
        db.session.commit()
        flash('تم إرسال رسالتك بنجاح!', 'success')
        return redirect(url_for('contact'))
    return render_template('contact.html', form=form)

# --- لوحة التحكم ---

@app.route('/admin/login', methods=['GET', 'POST'])
def admin_login():
    if current_user.is_authenticated:
        return redirect(url_for('admin_dashboard'))
    form = LoginForm()
    if form.validate_on_submit():
        user = User.query.filter_by(username=form.username.data).first()
        if user and user.check_password(form.password.data):
            login_user(user, remember=False)
            return redirect(url_for('admin_dashboard'))
        else:
            flash('اسم المستخدم أو كلمة المرور غير صحيحة.', 'danger')
    return render_template('admin/login.html', form=form)

@app.route('/admin/logout')
@login_required
def admin_logout():
    logout_user()
    return redirect(url_for('index'))

@app.route('/admin')
@login_required
def admin_dashboard():
    messages = Message.query.order_by(Message.id.desc()).limit(10).all()  # آخر 10 رسائل
    admins = User.query.all()
    return render_template('admin/dashboard.html', messages=messages, admins=admins)

# --- إدارة صور الهيدر ---
@app.route('/uploads/<path:filename>')
def uploaded_file(filename):
    return send_from_directory(os.path.join(app.static_folder, 'uploads'), filename)

@app.route('/admin/hero')
@login_required
def manage_hero():
    images = HeroImage.query.all()
    return render_template('admin/manage_hero.html', images=images)

@app.route('/admin/hero/add', methods=['GET', 'POST'])
@login_required
def add_hero():
    form = AddHeroImageForm()
    if form.validate_on_submit():
        image_path = save_image(form.image.data, 'hero')
        hero = HeroImage(
            filename=image_path,
            caption=form.caption.data,
            active=form.active.data
        )
        db.session.add(hero)
        db.session.commit()
        flash('تم إضافة صورة الهيدر!', 'success')
        return redirect(url_for('manage_hero'))
    return render_template('admin/add_hero.html', form=form)

@app.route('/admin/hero/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_hero(id):
    hero = HeroImage.query.get_or_404(id)
    form = EditHeroImageForm(obj=hero)
    if form.validate_on_submit():
        if form.image.data and form.image.data.filename:  
            if hero.filename:
                old_path = os.path.join('static', hero.filename)
                if os.path.exists(old_path):
                    os.remove(old_path)
            hero.filename = save_image(form.image.data, 'hero')
        hero.caption = form.caption.data
        hero.active = form.active.data
        db.session.commit()
        flash('تم تحديث الصورة!', 'success')
        return redirect(url_for('manage_hero'))
    return render_template('admin/edit_hero.html', form=form, hero=hero)

@app.route('/admin/hero/delete/<int:id>')
@login_required
def delete_hero(id):
    hero = HeroImage.query.get_or_404(id)

    if hero.filename:
        file_path = os.path.join(app.static_folder, hero.filename)
        if os.path.exists(file_path):
            os.remove(file_path)

    db.session.delete(hero)
    db.session.commit()

    flash('تم حذف الصورة نهائيًا.', 'info')
    return redirect(url_for('manage_hero'))

@app.route('/admin/gallery')
@login_required
def manage_gallery():
    images = GalleryImage.query.all()
    return render_template('admin/manage_gallery.html', images=images)

@app.route('/admin/gallery/add', methods=['GET', 'POST'])
@login_required
def add_gallery():
    form = GalleryImageForm()
    if form.validate_on_submit():
        image_path = save_image(form.image.data, 'gallery')
        gallery = GalleryImage(
            filename=image_path,
            caption=form.caption.data,
            active=form.active.data
        )
        db.session.add(gallery)
        db.session.commit()
        flash('تم إضافة صورة المعرض!', 'success')
        return redirect(url_for('manage_gallery'))
    return render_template('admin/add_gallery.html', form=form)

@app.route('/admin/gallery/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_gallery(id):
    gallery = GalleryImage.query.get_or_404(id)
    form = GalleryImageForm(obj=gallery)
    if form.validate_on_submit():
        if form.image.data and form.image.data.filename:
            if gallery.filename:
                old_path = os.path.join('static', gallery.filename)
                if os.path.exists(old_path):
                    os.remove(old_path)
            gallery.filename = save_image(form.image.data, 'gallery')
        gallery.caption = form.caption.data
        gallery.active = form.active.data
        db.session.commit()
        flash('تم تحديث الصورة!', 'success')
        return redirect(url_for('manage_gallery'))
    return render_template('admin/edit_gallery.html', form=form, gallery=gallery)

@app.route('/admin/gallery/delete/<int:id>')
@login_required
def delete_gallery(id):
    gallery = GalleryImage.query.get_or_404(id)

    if gallery.filename:
        file_path = os.path.join(app.static_folder, gallery.filename)
        if os.path.exists(file_path):
            os.remove(file_path)

    db.session.delete(gallery)
    db.session.commit()

    flash('تم حذف الصورة نهائيًا.', 'info')
    return redirect(url_for('manage_gallery'))

# --- إدارة المنتجات ---

@app.route('/admin/products')
@login_required
def manage_products():
    products = Product.query.all()
    brands = Brand.query.all()
    return render_template('admin/manage_products.html', products=products, brands=brands)

@app.route('/admin/products/add', methods=['GET', 'POST'])
@login_required
def add_product():
    form = ProductForm()
    brands = Brand.query.all()
    if form.validate_on_submit():
        image_path = save_image(form.image.data, 'products') if form.image.data else None
        product = Product(
            name=form.name.data,
            description=form.description.data,
            image=image_path,
            brand_id=form.brand_id.data
        )
        db.session.add(product)
        db.session.commit()
        flash('تم إضافة المنتج بنجاح!', 'success')
        return redirect(url_for('manage_products'))
    return render_template('admin/add_product.html', form=form, brands=brands)

@app.route('/admin/products/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_product(id):
    product = Product.query.get_or_404(id)
    form = ProductForm(obj=product)
    brands = Brand.query.all()

    if form.validate_on_submit():
        # تحديث بيانات المنتج
        product.name = form.name.data
        product.description = form.description.data
        product.brand_id = form.brand_id.data

        # التحقق من وجود ملف جديد قبل الحفظ
        if form.image.data and hasattr(form.image.data, 'filename') and form.image.data.filename != '':
            # حذف الصورة القديمة إذا كانت موجودة
            if product.image:
                old_path = os.path.join(app.static_folder, product.image)
                if os.path.exists(old_path):
                    os.remove(old_path)
            # حفظ الصورة الجديدة
            product.image = save_image(form.image.data, 'products')

        db.session.commit()
        flash('تم تحديث المنتج!', 'success')
        return redirect(url_for('manage_products'))

    return render_template('admin/edit_product.html', form=form, product=product, brands=brands)

@app.route('/admin/products/delete/<int:id>')
@login_required
def delete_product(id):
    product = Product.query.get_or_404(id)

    if product.image:
        file_path = os.path.join(app.static_folder, product.image)
        if os.path.exists(file_path):
            os.remove(file_path)

    db.session.delete(product)
    db.session.commit()
    
    flash('تم حذف المنتج نهائيًا.', 'info')
    return redirect(url_for('manage_products'))

# --- إدارة العلامات التجارية ---

@app.route('/admin/brands')
@login_required
def manage_brands():
    brands = Brand.query.all()
    return render_template('admin/manage_brands.html', brands=brands)

@app.route('/admin/brands/add', methods=['GET', 'POST'])
@login_required
def add_brand():
    form = BrandForm()
    if form.validate_on_submit():
        logo_path = save_image(form.logo.data, 'brands') if form.logo.data else None
        brand = Brand(name=form.name.data, logo=logo_path)
        db.session.add(brand)
        db.session.commit()
        flash('تمت إضافة العلامة التجارية!', 'success')
        return redirect(url_for('manage_brands'))
    return render_template('admin/add_brand.html', form=form)

@app.route('/admin/brands/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_brand(id):
    brand = Brand.query.get_or_404(id)
    form = BrandForm(obj=brand)

    if form.validate_on_submit():
        brand.name = form.name.data

        # التعامل مع الشعار بشكل آمن
        if form.logo.data and hasattr(form.logo.data, 'filename') and form.logo.data.filename != '':
            # حذف الشعار القديم إذا موجود
            if brand.logo:
                old_path = os.path.join('static', brand.logo)
                if os.path.exists(old_path):
                    os.remove(old_path)
            # حفظ الشعار الجديد
            brand.logo = save_image(form.logo.data, 'brands')

        db.session.commit()
        flash('تم تحديث العلامة!', 'success')
        return redirect(url_for('manage_brands'))

    return render_template('admin/edit_brand.html', form=form, brand=brand)

@app.route('/admin/brands/delete/<int:id>')
@login_required
def delete_brand(id):
    brand = Brand.query.get_or_404(id)

    if brand.logo:
        file_path = os.path.join(app.static_folder, brand.logo)
        if os.path.exists(file_path):
            os.remove(file_path)

    db.session.delete(brand)
    db.session.commit()
    
    flash('تم حذف العلامة التجارية نهائيًا.', 'info')
    return redirect(url_for('manage_brands'))

# --- إدارة الرسائل ---

@app.route('/admin/messages')
@login_required
def manage_messages():
    messages = Message.query.order_by(Message.id.desc()).all()
    return render_template('admin/messages.html', messages=messages)

@app.route('/admin/messages/read/<int:id>')
@login_required
def mark_message_read(id):
    msg = Message.query.get_or_404(id)
    msg.read = True
    db.session.commit()
    return redirect(url_for('manage_messages'))

@app.route('/admin/messages/delete/<int:id>')
@login_required
def delete_message(id):
    msg = Message.query.get_or_404(id)
    db.session.delete(msg)
    db.session.commit()
    flash('تم حذف الرسالة.', 'info')
    return redirect(url_for('manage_messages'))



# --- إدارة الفيديوهات ---
@app.route('/admin/videos')
@login_required
def manage_videos():
    videos = Video.query.all()
    return render_template('admin/manage_videos.html', videos=videos)


@app.route('/admin/videos/add', methods=['GET', 'POST'])
@login_required
def add_video():
    form = VideoForm()
    if form.validate_on_submit():
        result = save_video(form.video.data, 'videos')
        if result is None:
            flash("فشل رفع الفيديو.", "danger")
            return redirect(url_for('add_video'))
        
        video_filename, compression_msg = result

        video = Video(
            title=form.title.data,
            filename=video_filename,
            active=form.active.data
        )
        db.session.add(video)
        db.session.commit()

        flash('✅ تم إضافة الفيديو!', 'success')
        


        return redirect(url_for('manage_videos'))

    return render_template('admin/add_video.html', form=form)

@app.route('/admin/videos/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_video(id):
    video = Video.query.get_or_404(id)
    form = VideoForm(obj=video)

    if form.validate_on_submit():
        video.title = form.title.data
        video.active = form.active.data

        if form.video.data and form.video.data.filename:
            # حذف الفيديو القديم
            old_path = os.path.join(app.config['UPLOAD_FOLDER'], 'videos', video.filename)
            if os.path.exists(old_path):
                os.remove(old_path)

            # رفع وضغط الجديد
            result = save_video(form.video.data, 'videos')
            if result is None:
                flash("فشل رفع الفيديو الجديد.", "danger")
                return redirect(url_for('edit_video', id=id))
            
            video_filename, compression_msg = result
            video.filename = video_filename

            flash('✅ تم تحديث الفيديو!', 'success')
            if compression_msg:
                flash(compression_msg, 'info')

        db.session.commit()
        return redirect(url_for('manage_videos'))

    return render_template('admin/edit_video.html', form=form, video=video)

@app.route('/admin/videos/delete/<int:id>')
@login_required
def delete_video(id):
    video = Video.query.get_or_404(id)

    # حذف الملف من السيرفر
    if video.filename:
        file_path = os.path.join(app.config['UPLOAD_FOLDER'], 'videos', video.filename)
        if os.path.exists(file_path):
            os.remove(file_path)

    db.session.delete(video)
    db.session.commit()

    flash('🗑️ تم حذف الفيديو نهائيًا!', 'info')
    return redirect(url_for('manage_videos'))


@app.route('/admin/admins')
@login_required
def manage_admins():
    admins = User.query.all()
    return render_template('admin/manage_admins.html', admins=admins)

@app.route('/admin/admins/add', methods=['GET', 'POST'])
@login_required
def add_admin():
    form = AdminForm()
    if form.validate_on_submit():
        if User.query.filter_by(username=form.username.data).first():
            flash('اسم المستخدم موجود مسبقًا.', 'danger')
        else:
            admin = User(username=form.username.data)
            admin.set_password(form.password.data)
            db.session.add(admin)
            db.session.commit()
            flash('تم إضافة المسؤول بنجاح!', 'success')
            return redirect(url_for('admin_dashboard'))
    return render_template('admin/add_admin.html', form=form)

@app.route('/admin/admins/edit/<int:id>', methods=['GET', 'POST'])
@login_required
def edit_admin(id):
    if id == current_user.id:
        flash('لا يمكنك تعديل حسابك الحالي من هذه الصفحة.', 'warning')
        return redirect(url_for('admin_dashboard'))
    admin = User.query.get_or_404(id)
    form = AdminForm()
    if form.validate_on_submit():
        # التحقق من عدم تكرار اسم المستخدم
        existing = User.query.filter_by(username=form.username.data).first()
        if existing and existing.id != admin.id:
            flash('اسم المستخدم موجود مسبقًا.', 'danger')
        else:
            admin.username = form.username.data
            if form.password.data.strip():
                admin.set_password(form.password.data)
            db.session.commit()
            flash('تم تحديث المسؤول!', 'success')
            return redirect(url_for('admin_dashboard'))
    elif request.method == 'GET':
        form.username.data = admin.username
    return render_template('admin/edit_admin.html', form=form, admin=admin)

@app.route('/admin/admins/delete/<int:id>', methods=['POST'])
@login_required
def delete_admin(id):
    if id == current_user.id:
        flash('لا يمكنك حذف حسابك الحالي.', 'danger')
        return redirect(url_for('admin_dashboard'))
    admin = User.query.get_or_404(id)
    db.session.delete(admin)
    db.session.commit()
    flash('تم حذف المسؤول.', 'info')
    return redirect(url_for('admin_dashboard'))


# ----------------------------
# تشغيل التطبيق
# ----------------------------

if __name__ == '__main__':
    with app.app_context():
        db.create_all()
        if not User.query.filter_by(username='admin').first():
            admin = User(username='admin')
            admin.set_password('admin123')
            db.session.add(admin)
            db.session.commit()
            print("✅ تم إنشاء حساب إداري افتراضي: admin / admin123")
    app.run(debug=True)